#include "esp_camera.h"
#include "FS.h"
#include "SD_MMC.h"
#include "WiFi.h"
#include <WebServer.h>
#include <WiFiClient.h>
#include "time.h"

// === Пины камеры (AI-Thinker) ===
#define PWDN_GPIO_NUM     32
#define RESET_GPIO_NUM    -1
#define XCLK_GPIO_NUM      0
#define SIOD_GPIO_NUM     26
#define SIOC_GPIO_NUM     27
#define Y9_GPIO_NUM       35
#define Y8_GPIO_NUM       34
#define Y7_GPIO_NUM       39
#define Y6_GPIO_NUM       36
#define Y5_GPIO_NUM       21
#define Y4_GPIO_NUM       19
#define Y3_GPIO_NUM       18
#define Y2_GPIO_NUM        5
#define VSYNC_GPIO_NUM    25
#define HREF_GPIO_NUM     23
#define PCLK_GPIO_NUM     22

// === Настройки WiFi ===
const char* ssid = "PCwork";
const char* password = "123456789";

// === NTP ===
const char* ntpServer = "pool.ntp.org";
//const char* ntpServer = "time.windows.com";
const long  gmtOffset_sec = 10800; // GMT+3 для Москвы
const int   daylightOffset_sec = 0;

int numPhoto = 1;

bool ntp_on = true;

WebServer server(80);

// Переменные
bool autoCapture = false;
unsigned long lastCaptureTime = 0;
const unsigned long captureInterval = 5000; // 5 секунд между снимками

void setup() 
{
  Serial.begin(115200);

  pinMode(4, OUTPUT);
  digitalWrite(4, LOW); 

  Serial.setTimeout(5000);
  
  Serial.println("Initialization of the ESP32-CAM...");
  // Инициализация камеры
  if (!initCamera())  
  {
    Serial.println("ERROR: Camera initialization error!");
    while(1) delay(1000);
  }
  
  // Подключение к WiFi
  connectToWiFi();
  
  // Настройка времени через NTP
  setupNTP();
  
  // Инициализация SD карты
  if (!initSDCard()) 
  {
    Serial.println("ERROR: SD card initialization error!");
  }
  
  //  server.on("/", verAuth);

  // Настройка веб-сервера
  setupWebServer();
  
  Serial.println("The system is ready to work");
 
}

void loop() {
  server.handleClient();
  
  // Циклическая запись
  if (autoCapture && (millis() - lastCaptureTime > captureInterval)) {
    capturePhoto();
    lastCaptureTime = millis();
  }
  
  delay(100);
}

bool initCamera() 
{
  Serial.println("Initializing the camera...");
  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sscb_sda = SIOD_GPIO_NUM;
  config.pin_sscb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;
  config.xclk_freq_hz = 20000000;
  config.pixel_format = PIXFORMAT_JPEG; // +++
  
  if (psramFound()) 
  {
    // config.frame_size = FRAMESIZE_UXGA;
    config.frame_size = FRAMESIZE_QVGA; // или FRAMESIZE_CIF
    config.jpeg_quality = 10; // или 12
    config.fb_count = 2; // или даже 1 при нехватке памяти
    // Serial.println("PSRAM detected");
    // // Проверка PSRAM
    Serial.printf("PSRAM found: %d bytes\n", ESP.getPsramSize());
  }
  else 
  {
    config.frame_size = FRAMESIZE_SVGA;
    config.jpeg_quality = 12;
    config.fb_count = 1;
    Serial.println("No PSRAM detected");
  }
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) 
  {
    Serial.printf("ERROR: Camera initialization error: 0x%x", err);
    return false;
  }
  
  Serial.println("Camera initialized successfully");

  return true;
}

void connectToWiFi() 
{
  Serial.println("Connecting to WiFi...");
  Serial.printf("SSID: %s", ssid);
  
  WiFi.begin(ssid, password);
  
  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 20) {
    delay(500);
    Serial.print(".");
    attempts++;
  }
  
  if (WiFi.status() == WL_CONNECTED) 
  {
    Serial.println("WiFi is connected!");
    Serial.printf("IP address: %s", WiFi.localIP().toString().c_str());
    Serial.println("Power WiFi: " + String(WiFi.RSSI()) + " dBm\n");
  } 
  else 
  {
    Serial.println("\nERROR: WiFi connection error!\n"); 
  }
}

void setupNTP() // !!!
{
  Serial.println("Setting the time via NTP...");
  configTime(gmtOffset_sec, daylightOffset_sec, ntpServer);
  
  // Ждем получения времени
  struct tm timeinfo;
  if (!getLocalTime(&timeinfo)) 
  {
    Serial.println("ERROR: Couldn't get time from NTP");
    ntp_on = false;
    return;
  }
  char timeString[20];
  strftime(timeString, sizeof(timeString), "%Y-%m-%d %H:%M:%S", &timeinfo);
  Serial.printf("The time is set: %s\n", timeString);
}


bool initSDCard() 
{
  Serial.println("Initializing the SD card...");

  if (!SD_MMC.begin()) 
  {
    Serial.println("ERROR: SD card mounting error");
    return false;
  }
  
  uint8_t cardType = SD_MMC.cardType();
  if (cardType == CARD_NONE) 
  {
    Serial.println("ERROR: No SD card detected");
    return false;
  }
  uint64_t cardSize = SD_MMC.cardSize() / (1024 * 1024);
  uint64_t totalBytes = SD_MMC.totalBytes() / (1024 * 1024);
  uint64_t usedBytes = SD_MMC.usedBytes() / (1024 * 1024);
  float usagePercent = totalBytes > 0 ? (usedBytes * 100.0) / totalBytes : 0;
  Serial.println("SD card: " + String(cardSize) + " MB");
  Serial.println("Used: " + String(usedBytes) + " MB / " + String(totalBytes) + " MB");
  Serial.println("Completion: " + String(usagePercent, 1) + "%");
  return true;
}

void setupWebServer() 
{
 
  // Главная страница
  server.on("/", HTTP_GET, []() 
  {
    String html = createHTMLPage();
    server.send(200, "text/html", html);
  });

    // Сделать фото
  server.on("/capture", HTTP_GET, []() 
  {
    Serial.println("Command: Take a photo now");
    String filename = capturePhoto();
    if (filename != "") 
    {
      server.send(200, "text/plain", "✅ Фото сохранено: " + filename);
    } 
    else 
    {
      server.send(500, "text/plain", "❌ Ошибка сохранения фото");
    }
  });

  
  // Начать циклическую запись
  server.on("/start", HTTP_GET, []() 
  {
    Serial.println("Command: Start cyclic recording");
    autoCapture = true;
    lastCaptureTime = millis();
    server.send(200, "text/plain", "✅ Циклическая запись начата (интервал: " + String(captureInterval/1000) + "с)");
  });
  
  // Завершить циклическую запись
  server.on("/stop", HTTP_GET, []() {
    Serial.println("Command: End the loop recording");
    autoCapture = false;
    server.send(200, "text/plain", "✅ Циклическая запись остановлена");
  });
  
  // Удалить все файлы
  server.on("/delete", HTTP_GET, []() 
  {
    Serial.println("Command: Delete all files from SD");
    int deleted = deleteAllFiles();
    server.send(200, "text/plain", "✅ Удалено файлов: " + String(deleted));
  });
  
  server.begin();
  Serial.println("HTTP server is running on port 80");
}

String createHTMLPage() 
{
  String html = "<!DOCTYPE html><html><head>";
  html += "<meta charset='UTF-8'>";
  html += "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
  html += "<title>ESP32-CAM Control</title>";
  html += "<style>";
  html += "body{font-family:Arial,sans-serif;margin:20px;background:#f5f5f5;}";
  html += ".container{max-width:600px;margin:0 auto;background:white;padding:20px;border-radius:10px;box-shadow:0 2px 10px rgba(0,0,0,0.1);}";
  html += "h1{color:#333;text-align:center;}";
  html += "button{display:block;width:100%;padding:15px;margin:10px 0;border:none;border-radius:5px;font-size:16px;cursor:pointer;transition:background 0.3s;}";
  html += ".capture{background:#4CAF50;color:white;} .capture:hover{background:#45a049;}";
  html += ".start{background:#2196F3;color:white;} .start:hover{background:#0b7dda;}";
  html += ".stop{background:#ff9800;color:white;} .stop:hover{background:#e68900;}";
  html += ".delete{background:#f44336;color:white;} .delete:hover{background:#da190b;}";
  html += ".upload{background:#9C27B0;color:white;} .upload:hover{background:#7b1fa2;}";
  html += ".status{background:#607D8B;color:white;} .status:hover{background:#546e7a;}";
  html += ".status-box{background:#f8f9fa;padding:15px;border-radius:5px;margin-top:20px;font-family:monospace;font-size:12px;}";
  html += "</style></head><body>";
  html += "<div class='container'>";
  html += "<h1>📷 ESP32-CAM Control Panel</h1>";
  html += "<button class='capture' onclick='capturePhoto()'>📸 Сделать фото сейчас</button>";
  html += "<button class='start' onclick='startCapture()'>🔄 Начать циклическую запись</button>";
  html += "<button class='stop' onclick='stopCapture()'>⏹️ Завершить циклическую запись</button>";
  html += "<button class='delete' onclick='deleteFiles()'>🗑️ Удалить все файлы с SD</button>";
  html += "<div id='status' class='status-box'></div>";
  html += "<script>";
  html += "function capturePhoto(){fetch('/capture').then(r=>r.text()).then(alert);}";
  html += "function startCapture(){fetch('/start').then(r=>r.text()).then(alert);}";
  html += "function stopCapture(){fetch('/stop').then(r=>r.text()).then(alert);}";
  html += "function deleteFiles(){if(confirm('⚠️ Удалить ВСЕ файлы с SD карты?')){fetch('/delete').then(r=>r.text()).then(alert);}}";
  html += "function uploadFiles(){fetch('/upload').then(r=>r.text()).then(alert);}";
  // html += "function getStatus(){fetch('/status').then(r=>r.text()).then(t=>document.getElementById('status').innerHTML=t.replace(/\\n/g,'<br>'));}";
  html += "</script>";
  html += "</div></body></html>";
  return html;
}

int deleteAllFiles() 
{
  Serial.println("Start deleting all files...");
  fs::FS &fs = SD_MMC;
  File root = fs.open("/");
  
  int count = 0;
  File file = root.openNextFile();
  while (file) {
    String filename = String(file.name());
    if (!file.isDirectory()) 
    {
      if (fs.remove("//" + filename))
      {
        Serial.println("Deleted: " + filename);
        count++;
      } 
      else 
      {
        Serial.println("ERROR: Deletion error: " + filename);
      }
    }
    file = root.openNextFile();
  }
  
  Serial.printf("The deletion is complete. Total deleted: %d файлов\n", count);
  return count;
}

String capturePhoto() // !!!
{
  Serial.println("Capturing a photo...");

  camera_fb_t * fb = esp_camera_fb_get();
  if (!fb) 
  {
    Serial.println("ERROR: Camera capture error");
    return "";
  }
  Serial.println(getDateTimeString());
  String filename = "/" + getDateTimeString() + ".jpg";
  Serial.printf("Saving the file: %s\n", filename.c_str());
  
  fs::FS &fs = SD_MMC;
  File file = fs.open(filename.c_str(), FILE_WRITE);
  // File file = SD_MMC.open(filename.c_str(), FILE_WRITE);
  if (!file) 
  {
    Serial.println("ERROR: File creation error");
    esp_camera_fb_return(fb);
    return "";
  }
  bool saved = file.write(fb->buf, fb->len);
  file.close();
  esp_camera_fb_return(fb);
  
  if (saved) 
  {
    Serial.printf("Photo saved: %s (%d byte)\n", filename.c_str(), fb->len);
    return filename;
  } 
  else 
  {
    Serial.println("ERROR: File recording error");
    return "";
  }
}

String findFileNum(const char* extension) {
  File root = SD_MMC.open("/");
  if (!root) {
    Serial.println("Couldn't open the root directory");
    return "";
  }

  File file = root.openNextFile();
  while (file) {
    if (!file.isDirectory()) {
      String fileName = file.name();
      // Проверяем, оканчивается ли имя файла на нужное расширение
      if (fileName.endsWith(extension)) 
      {
        file.close();
        root.close();
        Serial.println("The file name with the num extension " + fileName);
        return fileName;
      }
    }
    file = root.openNextFile();
  }
  root.close();
  return ""; // Файл не найден
}

void writeFile(fs::FS &fs, const char *path, const char *message) 
{
  Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE);
  if (!file) {
    Serial.println("Failed to open file for writing");
    return;
  }
  if (file.print(message)) {
    Serial.println("File written");
  } else {
    Serial.println("Write failed");
  }
}


String getDateTimeString() // !!!
{
  if(!ntp_on)
  {
    String res = findFileNum(".num");
    if(res == "")
    {
      writeFile(SD_MMC, "/00001.num", "Hello ");
      return "00001";
    }
    else
    {
      int num = res.toInt();
      num++;
      char buffer[10];           // достаточно места под строку
      sprintf(buffer, "%05d", num);  // 5 цифр, с ведущими нулями → "00042"
      String str = String(buffer);
      String oldFile = "/" + res;
      String newFile = "/" + str + ".num";
      Serial.println("oldFile " + oldFile);
      Serial.println("newFile " + newFile);
      renameFile(SD_MMC, oldFile.c_str(), newFile.c_str());
      return str;
    }
  }
  struct tm timeinfo;
  getLocalTime(&timeinfo); // !!!
  char timeString[20];
  strftime(timeString, sizeof(timeString), "%Y%m%d_%H%M%S", &timeinfo);
  return String(timeString);
}


void renameFile(fs::FS &fs, const char *path1, const char *path2) 
{
  Serial.printf("Renaming file %s to %s\n", path1, path2);
  if (fs.rename(path1, path2)) {
    Serial.println("File renamed");
  } else {
    Serial.println("Rename failed");
  }
}